function trendchart(x,model,buffer)
%TRENDCHART makes a pls prediction and plots a running trend chart
% Plots of prediction, residuals, and T^2 are retained. A short
% term history feeds into a long term history at a given compression
% ratio (running mean of short-term history)
%IN: (newx)  new x-block to predict from
%    (model) PLS model structure
%    (buffer) optional vector describing length of short-term, buffer, 
%       and long-term histories. The buffer length defines the size of
%       the running-average window. (default = [70 3 70])
%I/O: trendchart(x,model,buffer)

%(c) COPYRIGHT Eigenvector Research, 2001
%9/01 jms

if nargin < 3;
  buffer = [70 3 70];
end

[yprdn,resn,tsqn] = modlpred(x,model,0);    %make prediction

persistent trend

if isempty(trend)
  trend.prdn   = [];
  trend.resn   = [];
  trend.tsqn   = [];
  trend.figure = gcf;
  %buffer = length of short history, buffer (ratio of short to long), and long history in data points
end

%add to short and shift into buffer and long history
trend.prdn = maketrend(trend.prdn,yprdn,buffer);
trend.resn = maketrend(trend.resn,resn,buffer);
trend.tsqn = maketrend(trend.tsqn,tsqn,buffer);

figure(trend.figure)
subplot(3,1,1);
plottrend(trend.prdn,[],buffer)
ylabel('Predicted Value')

subplot(3,1,2);
plottrend(trend.resn,model.reslim,buffer)
ylabel('Q Value')

subplot(3,1,3);
plottrend(trend.tsqn,model.tsqlim,buffer)
ylabel('T^2 Value')

%---------------------------------------
function trend = maketrend(trend,data,bufferlength)
%add data to short and shift oldest into buffer and maybe also add to long history

if isempty(trend);
  trend.short  = zeros(1,bufferlength(1))*nan;
  trend.buffer = zeros(1,bufferlength(2))*nan;
  trend.long   = zeros(1,bufferlength(3))*nan;
end

%add to short and shift into buffer and long history
while length(trend.short) >= bufferlength(1);
  trend.buffer(end+1) = trend.short(1);
  trend.short(1)      = [];
  if length(trend.buffer) >= bufferlength(2);
    trend.long(end+1) = mean(trend.buffer);
    if length(trend.long) >= bufferlength(3);
      trend.long = trend.long(end-bufferlength(3)+1:end);
    end
    trend.buffer = [];
  end
end

trend.short(end+1)  = data;

%---------------------------------------
function plottrend(trend,limits,buffer)

if nargin < 2 | isempty(limits);
  limits = [nan nan];
elseif length(limits) == 1;
  limits(2) = nan;
end

ool.short = (trend.short > limits(1) | trend.short < limits(2));
ool.long  = (trend.long  > limits(1) | trend.long  < limits(2));

x.short = [(length(trend.short)-1):-1:0];
x.long  = [(length(trend.long)-1):-1:0]+length(trend.short);

plot(x.long,trend.long,'b.-',x.long(ool.long),trend.long(ool.long),'r.',...
  x.short,trend.short,'b.-',x.short(ool.short),trend.short(ool.short),'r.')
hline(limits(1),'r--')
hline(limits(2),'r--')
vline((x.short(1)+x.long(end))/2,'g-')
set(gca,'xdir','reverse')

%Adjust labels to match buffer compression
ticks = get(gca,'xtick')';
ticks = (ticks-(buffer(1)*(ticks>=buffer(1)))).*((buffer(2)-1).*(ticks>=buffer(1))+1)+(buffer(1)*(ticks>=buffer(1)));
set(gca,'xticklabel',num2str(ticks));

